package org.msh.etbm.services.mobile.init;

import org.apache.commons.beanutils.PropertyUtils;
import org.jboss.seam.Component;
import org.msh.etbm.rest.mobile.SyncFromServerRequest;
import org.msh.etbm.services.mobile.InitAndSyncDataGenerator;
import org.msh.etbm.services.mobile.model.*;
import org.msh.tb.entities.Tbunit;
import org.msh.tb.entities.enums.*;
import org.msh.tb.ng.entities.enums.Qualification;
import org.msh.utils.date.Period;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Created by rmemoria on 24/7/17.
 */
public class Step5InitGenerator implements InitAndSyncDataGenerator<CaseData2Response> {

    @Override
    public CaseData2Response generate(Tbunit unit) {
        return generate(unit, null);
    }

    @Override
    public CaseData2Response generate(Tbunit unit, SyncFromServerRequest req) {
        CaseData2Response resp = new CaseData2Response();

        resp.setExamsDST(loadDSTExams(unit, req));

        resp.setMedicalExaminations(loadMedicalExaminations(unit, req));

        resp.setHealthUnits(loadTreatmentHealthUnit(unit, req));

        resp.setPrescribedMedicines(loadPrescribedMedicines(unit, req));

        resp.setContacts(loadContacts(unit, req));

        resp.setExamsXRay(loadExamsXRay(unit, req));

        resp.setCaseSideEffects(loadCaseSideEffects(unit, req));

        resp.setComorbidities(loadComorbidities(unit, req));

        resp.setPrevTBTreatments(loadPrevTBTreatments(unit, req));

        return resp;
    }

    private EntityManager getEntityManager() {
        return (EntityManager) Component.getInstance("entityManager");
    }


    /**
     * Return the list of medical examinations to be sent to the client
     * @param unit
     * @return
     */
    private List<MedicalExaminationData> loadMedicalExaminations(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select e.id, a.id, e.date, e.appointmentType, " +
                "e.usingPrescMedicines, e.height, e.weight, e.nutrtnSupport, e.usingPrescMedicines, " +
                "e.reasonNotUsingPrescMedicines, e.responsible, e.qualification, e.otherQualifiedProfessional, " +
                "e.patientRefBy, e.referredByUnitName, e.refByDate, e.patientReferred, " +
                "e.referredToUnitName, e.sideeffect.id, e.refToDate, e.comments " +
                "from MedicalExaminationNG e " +
                "join e.tbcase a " +
                "where a.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<MedicalExaminationData> res = new ArrayList<MedicalExaminationData>(lst.size());

        for (Object[] vals: lst) {
            MedicalExaminationData data = new MedicalExaminationData();
            data.setServerId((Integer)vals[0]);
            data.setCaseId((Integer)vals[1]);
            data.setDate((Date)vals[2]);
            data.setAppointmentType((MedAppointmentType)vals[3]);
            YesNoType yesNo = (YesNoType)vals[4];
            data.setUsingPrescMedicines(yesNo != null ? yesNo.isYes() : null);
            data.setHeight(vals[5] != null ? ((Number)vals[5]).doubleValue() : null);
            data.setWeight(vals[6] != null ? ((Number)vals[6]).doubleValue() : null);
            yesNo = (YesNoType)vals[7];
            data.setNutrtnSupport(yesNo != null ? yesNo.isYes() : null);
            yesNo = (YesNoType)vals[8];
            data.setUsingPrescMedicines(yesNo != null ? yesNo.isYes() : null);
            data.setReasonNotUsingPrescMedicines((String)vals[9]);
            data.setResponsible((String)vals[10]);
            data.setQualification((Qualification)vals[11]);
            data.setOtherQualifiedProfessional((String)vals[12]);
            data.setPatientRefBy((ReferredBy)vals[13]);
            data.setReferredByUnitName((String)vals[14]);
            data.setRefByDate((Date)vals[15]);
            yesNo = (YesNoType)vals[16];
            data.setPatientReferred(yesNo != null ? yesNo.isYes() : null);
            data.setReferredByUnitName((String)vals[17]);
            data.setSideEffectId((Integer)vals[18]);
            data.setRefToDate((Date)vals[19]);
            data.setComments((String)vals[20]);

            res.add(data);
        }

        return res;
    }

    /**
     * Fill DST data
     * @param unit
     * @return
     */
    private List<ExamDSTData> loadDSTExams(Tbunit unit, SyncFromServerRequest req) {
        EntityManager em = getEntityManager();

        String qryString = "select a.id, a.tbcase.id, a.dateCollected," +
                "a.dateRelease, a.comments, a.method.id, a.sampleNumber, " +
                "r.substance.abbrevName.name1, r.result, a.laboratory.id " +
                "from ExamDSTResult r join r.exam a " +
                "where a.tbcase.ownerUnit.id = :id $CONDITION " +
                "order by a.id";

        if (req != null && req.hasBothIndexes()) {
            String cond = " and a.lastTransaction.id > " + req.getServerIndexFrom();
            cond += " and a.lastTransaction.id <= " + req.getServerIndexTo();
            qryString = qryString.replace("$CONDITION", cond);
        } else {
            qryString = qryString.replace("$CONDITION", "");
        }

        List<Object[]> lst = em
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<ExamDSTData> res = new ArrayList<ExamDSTData>();
        int prevId = 0;
        ExamDSTData data = null;
        for (Object[] vals: lst) {
            int id = (Integer)vals[0];

            // if is a new exam, create a new object to receive the data
            if (id != prevId) {
                data = new ExamDSTData();
                res.add(data);

                data.setServerId(id);
                data.setCaseId((Integer)vals[1]);
                data.setDate((Date)vals[2]);
                data.setDateRelease((Date)vals[3]);
                data.setComments((String)vals[4]);
                data.setMethod((Integer)vals[5]);
                data.setSampleNumber((String)vals[6]);
                data.setLaboratoryId((Integer)vals[9]);
                prevId = id;
            }

            DstResult dstres = (DstResult)vals[8];
            setDstResult(data, (String)vals[7], dstres);
        }

        return res;
    }

    public void setDstResult(ExamDSTData data, String substanceName, DstResult result) {
        substanceName = substanceName.replace("/", "");
        String propName = "result" + substanceName;

        try {
            PropertyUtils.setProperty(data, propName, result);
        } catch (Exception e) {
            System.out.println("Property not found in " + ExamDSTData.class.getCanonicalName() +
                    " = " + propName);
        }
    }

    /**
     * Load the treatment health unit of the cases from the given unit
     * @param unit
     * @return
     */
    private List<TreatmentHealthUnitData> loadTreatmentHealthUnit(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select hu.id, hu.tbcase.id, hu.period, hu.tbunit.id " +
                "from TreatmentHealthUnit hu " +
                "where hu.tbunit.id = :id " +
                "and hu.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and hu.tbcase.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and hu.tbcase.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<TreatmentHealthUnitData> res = new ArrayList<TreatmentHealthUnitData>(lst.size());

        for (Object[] val: lst) {
            TreatmentHealthUnitData data = new TreatmentHealthUnitData();
            data.setId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            Period p = (Period)val[2];
            if (p != null) {
                data.setIniDate(p.getIniDate());
                data.setEndDate(p.getEndDate());
            }
            data.setTbunitId((Integer)val[3]);
            res.add(data);
        }

        return res;
    }

    /**
     * Load the prescribed medicines of the cases from the given unit
     * @param unit
     * @return
     */
    private List<PrescribedMedicineData> loadPrescribedMedicines(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select pm.id, pm.tbcase.id, pm.period, pm.medicine.id," +
                "pm.comments, pm.doseUnit, pm.frequency, pm.source.id, pm.tbcase.iniContinuousPhase " +
                "from PrescribedMedicine pm " +
                "where pm.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and pm.tbcase.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and pm.tbcase.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<PrescribedMedicineData> res = new ArrayList<PrescribedMedicineData>(lst.size());

        for (Object[] val: lst) {
            PrescribedMedicineData data = new PrescribedMedicineData();
            data.setId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            Period p = (Period)val[2];
            data.setIniDate(p.getIniDate());
            data.setEndDate(p.getEndDate());
            data.setMedicineId((Integer)val[3]);
            data.setComments((String)val[4]);
            data.setDoseUnit((Integer)val[5]);
            data.setFrequency((Integer)val[6]);
            data.setSourceId((Integer)val[7]);
            Date iniContinuousPhase = (Date)val[8];
            data.setContinuationPhase(p.getIniDate().compareTo(iniContinuousPhase) >= 0);

            res.add(data);
        }
        return res;
    }

    /**
     * Load the list of contacts from the cases from the given unit
     * @param unit
     * @return
     */
    private List<ContactData> loadContacts(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select c.id, c.tbcase.id, c.age, c.name, c.contactOtherName, " +
                "c.contactLastName, c.gender, c.contactType.id, c.examinated, c.dateOfExamination, " +
                "c.hasTB, c.startedAntiTB, c.eligibleForINH, c.commencedOnINH, c.conduct.id, c.comments " +
                "from TbContactNG c " +
                "where c.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and c.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and c.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<ContactData> res = new ArrayList<ContactData>(lst.size());

        for (Object[] val: lst) {
            ContactData data = new ContactData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setAge((String)val[2]);
            data.setFirstName((String)val[3]);
            data.setOtherName((String)val[4]);
            data.setLastName((String)val[5]);
            data.setGender((Gender)val[6]);
            data.setContactTypeId((Integer)val[7]);
            data.setExaminated((Boolean)val[8]);
            data.setDateOfExamination((Date)val[9]);
            data.setHasTB((Boolean)val[10]);
            data.setStartedAntiTB((Boolean)val[11]);
            data.setEligibleForINH((Boolean)val[12]);
            data.setCommencedOnINH((Boolean)val[13]);
            data.setConductId((Integer)val[14]);
            data.setComments((String)val[15]);
            res.add(data);
        }

        return res;
    }

    /**
     * Load the list of exams XRay from the cases of the given unit
     * @param unit
     * @return
     */
    private List<ExamXRayData> loadExamsXRay(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select e.id, e.tbcase.id, e.date, e.evolution, " +
                "e.presentation.id, e.comments " +
                "from ExamXRay e " +
                "where e.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<ExamXRayData> res = new ArrayList<ExamXRayData>(lst.size());
        for (Object[] val: lst) {
            ExamXRayData data = new ExamXRayData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setDate((Date)val[2]);
            data.setEvolution((XRayEvolution) val[3]);
            data.setPresentation((Integer)val[4]);
            data.setComments((String)val[5]);
            res.add(data);
        }
        return res;
    }

    /**
     * Load the list of case side effects from the cases from the given unit
     * @param unit
     * @return
     */
    private List<CaseSideEffectData> loadCaseSideEffects(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select c.id, c.tbcase.id, c.sideEffect.value.id, c.substance.id, " +
                "c.substance2.id, c.month, c.comment " +
                "from CaseSideEffect c " +
                "where c.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and c.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and c.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<CaseSideEffectData> res = new ArrayList<CaseSideEffectData>(lst.size());

        for (Object[] val: lst) {
            CaseSideEffectData data = new CaseSideEffectData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setSideEffect((Integer)val[2]);
            data.setSubstance((Integer)val[3]);
            data.setSubstance2((Integer)val[4]);
            data.setMonth((Integer)val[5]);
            data.setComments((String)val[6]);
            res.add(data);
        }

        return res;
    }

    /**
     * Load the list of case comorbidities from the cases from the given unit
     * @param unit
     * @return
     */
    private List<CaseComorbidityData> loadComorbidities(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select c.id, c.tbcase.id, c.comorbidity.id, c.duration, c.comment " +
                "from CaseComorbidity c " +
                "where c.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and c.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and c.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<CaseComorbidityData> res = new ArrayList<CaseComorbidityData>(lst.size());

        for (Object[] val: lst) {
            CaseComorbidityData data = new CaseComorbidityData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setComorbidity((Integer)val[2]);
            data.setDuration((String)val[3]);
            data.setComment((String)val[4]);
            res.add(data);
        }

        return res;
    }

    /**
     * Load the list of case previous tb treatments from the cases from the given unit
     * @param unit
     * @return
     */
    private List<PrevTBTreatmentData> loadPrevTBTreatments(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select p.id, p.tbcase.id, p.month, p.year," +
                "p.outcome, s.abbrevName.name1 " +
                "from PrevTBTreatment p join p.substances s " +
                "where p.tbcase.ownerUnit.id = :id $CONDITION " +
                "order by p.id";

        if (req != null && req.hasBothIndexes()) {
            String cond = " and p.tbcase.lastTransaction.id > " + req.getServerIndexFrom();
            cond += " and p.tbcase.lastTransaction.id <= " + req.getServerIndexTo();
            qryString = qryString.replace("$CONDITION", cond);
        } else {
            qryString = qryString.replace("$CONDITION", "");
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<PrevTBTreatmentData> res = new ArrayList<PrevTBTreatmentData>();
        int prevId = 0;
        PrevTBTreatmentData data = null;
        for (Object[] vals: lst) {
            int id = (Integer)vals[0];

            // if is a new prev treat, create a new object to receive the data
            if (id != prevId) {
                data = new PrevTBTreatmentData();
                res.add(data);

                data.setId(id);
                data.setCaseId((Integer)vals[1]);
                data.setMonth((Integer)vals[2]);
                data.setYear((Integer)vals[3]);
                data.setOutcome((PrevTBTreatmentOutcome)vals[4]);

                prevId = id;
            }

            setPrevTreatSubstance(data, (String)vals[5]);
        }

        return res;
    }

    public void setPrevTreatSubstance(PrevTBTreatmentData data, String substanceName) {
        substanceName = substanceName.replace("/", "");
        String propName = "has" + substanceName;

        try {
            PropertyUtils.setProperty(data, propName, true);
        } catch (Exception e) {
            System.out.println("Property not found in " + ExamDSTData.class.getCanonicalName() +
                    " = " + propName);
        }
    }
}
